<?php

namespace App\classes\service;

use Exception;
use App\Models\User;
use App\Models\Reseller;
use App\Models\UserAddress;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\DB;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rules\Password;
use Illuminate\Validation\Validator as ValidatorReturn;

class UserService
{
  public function Index(Request $request)
  {
    $users =  User::paginate(15);
    return $users;
  }
  public function Create(Request $request)
  {
    DB::beginTransaction();
    try {
      $reseller = Reseller::find($request->reseller_id);
      $user = User::create([
        "country_id"  => $request->country_id,
        "reseller"    => $reseller?->name,
        "name"        => trim(htmlspecialchars($request->name)),
        "email"       => trim(htmlspecialchars($request->email)),
        "address"     => trim(htmlspecialchars($request->address_1)),
        "telephone"   => trim(htmlspecialchars($request->telephone)),
        "password"    => Hash::make(trim($request->password)),
        "status"      => "active",
      ]);
      $address_2  = trim(htmlspecialchars($request->address_2));
      $state      = trim(htmlspecialchars($request->state));
      $city      = trim(htmlspecialchars($request->city));
      $post_code  = trim(htmlspecialchars($request->post_code));
      UserAddress::create([
        "user_id"   => $user->id,
        "address_1" => trim(htmlspecialchars($request->address_1)),
        "address_2" => $address_2 === "" ? null : $address_2,
        "state"     => $state === "" ? null : $state,
        "city"      => $city === "" ? null : $city,
        "street"    => trim(htmlspecialchars($request->street)),
        "post_code" => $post_code === "" ? null : $post_code,
      ]);
      $user->assignRole(Role::where("name", "user")->first());
      DB::commit();
      return $user;
    } catch (Exception $e) {
      DB::rollBack();
      return back()->withInput()->with("error", "An error occured while saving user, please try again !!");
    }
  }
  public function Update(Request $request, User $user): User
  {
    $reseller = Reseller::find($request->reseller_id);
    $user->update([
      "country_id"  => $request->country_id,
      "reseller"    => $reseller?->name,
      "name"        => trim(htmlspecialchars($request->name)),
      "email"       => trim(htmlspecialchars($request->email)),
      "address"     => trim(htmlspecialchars($request->address_1)),
      "telephone"   => trim(htmlspecialchars($request->telephone)),
      "status"    => $request->status,
    ]);
    return $user;
  }
  public function ChangePassword(Request $request, User $user): User
  {
    $user->update(["password" => Hash::make(trim($request->password)),]);
    return $user;
  }

  public function Delete(User $user): bool
  {
    return $user->delete();
  }
  /**
   * Validation
   *
   * @param  Request $request
   * @param  string $method
   * @param  User|bool $user
   * @return ValidatorReturn|null
   */
  public function DataValidation(Request $request, String $method, User|bool $user = null, $pswd = false): ValidatorReturn|null
  {
    if ($pswd) {
      return Validator::make($request->all(), [
        "password"  => ["required", Password::min(8)->numbers()->letters()->mixedCase()],
      ]);
    }
    switch (strtolower($method)) {
      case 'post':
        return Validator::make($request->all(), [
          "name"        => ["required", "string"],
          "email"       => ["required", "unique:users,email"],
          "telephone"   => ["required", "string"],
          "address_1"   => ["required", "string"],
          "address_2"   => ["nullable", "string"],
          "state"       => ["nullable", "string"],
          "city"        => ["nullable", "string"],
          "street"      => ["required", "string"],
          "post_code"   => ["nullable", "string"],
          "country_id"  => ["required", "exists:countries,id"],
          "reseller_id" => ["nullable", "exists:resellers,id"],
          "password"    => ["required", Password::min(8)->numbers()->letters()->mixedCase()],
        ]);
      case 'patch':
        return Validator::make($request->all(), [
          "name"        => ["required", "string"],
          "email"       => ["required", Rule::unique("users", "email")->ignore($user->id)],
          "telephone"   => ["required", "string"],
          "address_1"   => ["required", "string"],
          // "address_2"   => ["nullable", "string"],
          // "state"       => ["nullable", "string"],
          // "city"        => ["nullable", "string"],
          // "street"      => ["required", "string"],
          // "post_code"   => ["nullable", "string"],
          "country_id"  => ["required", "exists:countries,id"],
          "reseller_id" => ["nullable", "exists:resellers,id"],
          "status"      => ["required", "exists:statuses,name"],
        ]);
      default:
        return null;
    }
  }
}
